module Views.QRImageView exposing (qrCodeView)

import Css exposing (..)
import Html.Styled as Html exposing (Html, div, fromUnstyled, text)
import Html.Styled.Attributes exposing (css)
import Html.Styled.Events exposing (onClick)
import MediaQueries exposing (withMediaDesktop)
import Model exposing (Model, CenterContent(..), CenterTextConfig, CenterImageConfig)
import QRCode
import QRTypes exposing (encodeQRType)
import State exposing (Msg(..))
import Svg.Styled as Svg exposing (Svg)
import Svg.Styled.Attributes as SvgAttrs
import Views.PrimitiveComponents exposing (styledGroup)


qrCodeView : Model -> Html Msg
qrCodeView model =
    styledGroup
        [ css
            [ flexShrink (int 0)
            , maxWidth (vw 100)  -- Allow full width for QR code
            , padding (px 10)    -- Add some outer padding
            , withMediaDesktop
                [ marginLeft (px 6)
                , maxWidth (px 700)  -- Increase max width on desktop
                ]
            ]
        ]
        [ qrCodeImage model
        , Html.button
            [ onClick DownloadQRCodeWithOverlays, css [ marginTop (px 12), display block, width (pct 100) ] ]
            [ text "Download PNG" ]
        ]


qrCodeImage : Model -> Html Msg
qrCodeImage model =
    case QRCode.fromStringWith model.errorCorrection (encodeQRType model.qrType) of
        Err _ ->
            text "Something went wrong"

        Ok code ->
            div
                [ css
                    [ -- QR code container with extra space to prevent cutoff
                      minWidth (px (toFloat model.qrStyle.size + 60))  -- Extra space for QR code
                    , minHeight (px (toFloat model.qrStyle.size + 60)) -- Extra space for QR code
                    , maxWidth (vw 85)  -- Increase max width
                    , maxHeight (vh 70) -- Increase max height
                    , position relative
                    , margin auto
                    , display block
                    , padding (px 30)  -- Increase padding for better quiet zone
                    , backgroundColor (hex "ffffff")  -- White background for quiet zone
                    , border3 (px 1) solid (hex "f0f0f0")  -- Subtle border
                    , borderRadius (px 8)  -- Slightly more rounded
                    , boxShadow4 (px 0) (px 4) (px 12) (rgba 0 0 0 0.15)  -- Better shadow
                    , overflow visible  -- Ensure content isn't clipped
                    , boxSizing contentBox  -- Don't include padding in size calculations
                    , if model.qrStyle.backgroundColor == "transparent" then
                        batch
                            [ backgroundImage (url "data:image/svg+xml;charset=utf-8,%3csvg width='20' height='20' xmlns='http://www.w3.org/2000/svg'%3e%3cpath d='M0 0h10v10H0zm10 10h10v10H10z' fill='%23f0f0f0'/%3e%3c/svg%3e")
                            , backgroundSize (px 20)
                            ]
                      else
                        batch []
                    ]
                ]
                [ customQRCodeSvg model code
                , centerOverlay model
                ]


customQRCodeSvg : Model -> QRCode.QRCode -> Html Msg
customQRCodeSvg model code =
    let
        matrix =
            QRCode.toMatrix code

        size =
            List.length matrix

        moduleSize =
            toFloat model.qrStyle.size / toFloat size

        modules =
            matrix
                |> List.indexedMap
                    (\row rowData ->
                        rowData
                            |> List.indexedMap
                                (\col isDark ->
                                    if isDark then
                                        roundedRect
                                            (col * Basics.round moduleSize)
                                            (row * Basics.round moduleSize)
                                            (Basics.round moduleSize)
                                            model.qrStyle.cornerRadius
                                            model.qrStyle.moduleScale
                                            model.qrStyle.foregroundColor

                                    else
                                        Svg.g [] []
                                )
                    )
                |> List.concat
    in
    Svg.svg
        [ SvgAttrs.width (String.fromInt model.qrStyle.size)
        , SvgAttrs.height (String.fromInt model.qrStyle.size)
        , SvgAttrs.viewBox ("0 0 " ++ String.fromInt model.qrStyle.size ++ " " ++ String.fromInt model.qrStyle.size)
        , SvgAttrs.css [ width (px (toFloat model.qrStyle.size)), height (px (toFloat model.qrStyle.size)), display block ]
        ]
            (if model.qrStyle.backgroundColor == "transparent" then
                modules
             else
                [ Svg.rect
                    [ SvgAttrs.width "100%"
                    , SvgAttrs.height "100%"
                    , SvgAttrs.fill model.qrStyle.backgroundColor
                    ]
                    []
                ]
                    ++ modules
            )


roundedRect : Int -> Int -> Int -> Float -> Float -> String -> Svg msg
roundedRect x y size radius scale color =
    let
        scaledSize = toFloat size * scale
        offset = (toFloat size - scaledSize) / 2.0
        scaledX = toFloat x + offset
        scaledY = toFloat y + offset
    in
    Svg.rect
        [ SvgAttrs.x (String.fromFloat scaledX)
        , SvgAttrs.y (String.fromFloat scaledY)
        , SvgAttrs.width (String.fromFloat scaledSize)
        , SvgAttrs.height (String.fromFloat scaledSize)
        , SvgAttrs.rx (String.fromFloat radius)
        , SvgAttrs.ry (String.fromFloat radius)
        , SvgAttrs.fill color
        ]
        []


centerOverlay : Model -> Html Msg
centerOverlay model =
    case model.qrStyle.centerContent of
        NoCenterContent ->
            text ""

        CenterImage config ->
            if String.isEmpty config.url then
                text ""

            else
                Html.img
                    [ Html.Styled.Attributes.src config.url
                    , css
                        [ position absolute
                        , top (pct 50)
                        , left (pct 50)
                        , transform (translate2 (pct -50) (pct -50))
                        , maxWidth (pct (toFloat config.size))
                        , maxHeight (pct (toFloat config.size))
                        , borderRadius (px config.borderRadius)
                        , backgroundColor (hex (String.dropLeft 1 config.backgroundColor))
                        , padding (px 6)
                        , boxShadow4 (px 0) (px 2) (px 8) (rgba 0 0 0 0.15)
                        , property "object-fit" "contain"
                        , property "image-rendering" "crisp-edges"
                        , border3 (px (toFloat config.borderWidth)) solid (hex (String.dropLeft 1 config.borderColor))
                        ]
                    , Html.Styled.Attributes.alt "QR Center Image"
                    ]
                    []

        CenterText config ->
            if String.isEmpty config.text then
                text ""

            else
                Html.div
                    [ css
                        [ position absolute
                        , top (pct 50)
                        , left (pct 50)
                        , transform (translate2 (pct -50) (pct -50))
                        , backgroundColor (hex (String.dropLeft 1 config.backgroundColor))
                        , color (hex (String.dropLeft 1 config.textColor))
                        , padding (px (toFloat config.padding))
                        , borderRadius (px config.borderRadius)
                        , fontSize (px (toFloat config.fontSize))
                        , fontFamilies [ config.fontFamily ]
                        , property "font-weight" config.fontWeight
                        , property "font-style" config.fontStyle
                        , textAlign center
                        , whiteSpace noWrap
                        , boxShadow4 (px 0) (px 2) (px 4) (rgba 0 0 0 0.1)
                        , border3 (px (toFloat config.borderWidth)) solid (hex (String.dropLeft 1 config.borderColor))
                        ]
                    ]
                    [ text config.text ]


